# 📘 README – Mini Blockchain éducative “CryptoGamie”

> *Projet pédagogique réalisé en PHP / SQLite*  
> *Inspiré du fonctionnement des vraies blockchains (Bitcoin, Ethereum…)*  
> *Objectif : comprendre les principes sans avoir besoin d’un serveur complexe.*

---

## 🧱 1. Principe général d’une blockchain

Une **blockchain** est un grand **registre de transactions** partagé entre plusieurs ordinateurs (appelés *nœuds*).  
Chaque transaction est enregistrée dans un **bloc**, et chaque bloc est **lié cryptographiquement** au précédent, formant ainsi une *chaîne de blocs*.

> ⚙️ **Blockchain = base de données distribuée + sécurisée + transparente**

---

### 🧩 Les composants de base

| Élément | Description |
|----------|-------------|
| **Transaction** | Mouvement d’une valeur : “Alice envoie 5 Ƀ à Bob”. Chaque transaction est *signée* par l’émetteur avec une clé privée. |
| **Bloc** | Regroupe plusieurs transactions, possède une date, un lien vers le bloc précédent (`prev_hash`) et un *hash* unique calculé par une preuve de travail. |
| **Hash** | Empreinte numérique (fonction SHA-256) calculée à partir du contenu du bloc. Si on modifie quoi que ce soit → le hash change ! |
| **PoW (Proof of Work)** | “Preuve de travail” : le mineur doit trouver un nombre (`nonce`) tel que le hash commence par plusieurs zéros (ex. `0000a3f2…`). |
| **Consensus** | Règle qui permet à tous les ordinateurs d’avoir la même version de la chaîne. La règle classique : *“la chaîne la plus longue et valide gagne”*. |
| **Pair (peer)** | Autre serveur connecté au réseau qui échange des transactions et des blocs. |

---

### 🔐 Les clés et signatures

Chaque utilisateur possède :
- une **clé privée** (personnelle, pour signer)  
- une **clé publique** (partagée avec tous)  
- une **adresse** (hash de la clé publique, pour identifier le compte)

Lorsqu’un utilisateur signe une transaction, il prouve que **c’est bien lui** qui en est l’auteur, sans révéler sa clé privée.

---

## 🧭 2. Fonctionnement du projet *CryptoGamie*

CryptoGamie est une mini blockchain écrite en **PHP** avec une base de données **SQLite**.  
Chaque instance (ordinateur ou serveur) agit comme un **nœud** du réseau.

---

### ⚙️ Structure technique

| Fichier | Rôle |
|----------|------|
| `config.php` | Paramètres globaux (difficulté PoW, timezone, CORS) |
| `db.php` | Connexion SQLite et création automatique des tables |
| `crypto.php` | Gestion des clés RSA, signatures et adresses |
| `auth.php` | Création et connexion des utilisateurs (sessions PHP) |
| `chain.php` | Cœur de la blockchain : transactions, blocs, minage, consensus |
| `api.php` | Interface P2P (JSON) pour échanger entre serveurs |
| `index.php` | Interface web (Bootstrap) pour interagir avec le système |

---

## 🧩 3. Les étapes de fonctionnement

### 1️⃣ Création d’un compte

- L’utilisateur choisit un **nom d’utilisateur** et un **mot de passe**.  
- Le serveur génère automatiquement une **clé publique / clé privée RSA**.  
- L’adresse du compte est dérivée du hash de la clé publique.  
- Cette adresse servira à envoyer ou recevoir des transactions.

> Exemple :  
> `3fbd9a18f174e5b6c01e0b31d4a6a7b1234b56ac`

---

### 2️⃣ Création d’une transaction

- L’utilisateur connecté indique :
  - un **destinataire** (adresse)
  - un **montant**
- Le serveur construit un **payload** :  
  `adresse_émetteur | adresse_destinataire | montant | date`
- Ce texte est **signé** avec la clé privée de l’utilisateur.
- La transaction est ajoutée à la **file d’attente (mempool)**.

> 💡 Tant qu’un bloc n’est pas miné, les transactions restent “en attente”.

---

### 3️⃣ Minage (Proof of Work)

Lorsqu’on clique sur **« Miner un bloc »** :
1. Toutes les transactions en attente sont regroupées.  
2. Le serveur cherche un `nonce` tel que le **hash** du bloc commence par plusieurs zéros (selon la difficulté).  
3. Une fois trouvé :
   - le bloc est ajouté à la base,  
   - les transactions sont supprimées de la file d’attente.

> Exemple de hash valide avec difficulté 3 :  
> `000ab84e42f7c9230d5e47e9b9...`

---

### 4️⃣ Vérification d’intégrité

Chaque bloc stocke :
- le hash du bloc précédent (`prev_hash`)  
- son propre hash calculé avec le contenu complet  
- les signatures de toutes les transactions  

À tout moment, le système peut vérifier :
- que les blocs sont bien chaînés,  
- que tous les hash sont valides,  
- que chaque signature correspond bien à la clé publique de l’émetteur.

---

### 5️⃣ P2P (pair à pair)

Chaque serveur peut être connecté à d’autres *pairs*.  
Les échanges se font via des appels HTTP JSON (`api.php`).

#### Endpoints disponibles

| Route | Rôle |
|-------|------|
| `/api.php?route=chain` | Retourne la blockchain complète |
| `/api.php?route=pool` | Retourne les transactions en attente |
| `/api.php?route=tx_new` | Ajoute une nouvelle transaction reçue d’un pair |
| `/api.php?route=peers` | Liste des pairs connus |

#### Exemple de transaction reçue

```json
POST https://monserveur/crypto/api.php?route=tx_new
{
  "sender": "3fbd9a18...",
  "receiver": "1a4c0e...",
  "amount": 10,
  "pubkey": "-----BEGIN PUBLIC KEY-----...",
  "signature": "MEUCIQD3...",
  "created_at": "2025-11-01 15:30:12"
}
```

---

### 6️⃣ Consensus (chaîne la plus longue)

Quand plusieurs serveurs ont des versions différentes de la chaîne :
- chacun compare sa longueur avec celle des autres,  
- si un pair possède une chaîne **plus longue et valide**,  
  → elle est téléchargée et remplace la chaîne locale.

> ⚖️ Règle : *La chaîne la plus longue = celle qui a le plus de travail effectué.*

---

## 💡 4. Interface utilisateur

L’interface web (Bootstrap) est divisée en trois zones :

### 🧑 Compte
- Créer un compte ou se connecter  
- Voir son adresse publique et sa clé  
- Se déconnecter  

### 💸 Transactions
- Ajouter une transaction (adresse + montant)  
- Voir la file d’attente (*pool*)  
- Miner un bloc (*Proof of Work*)  
- Diffuser les transactions aux pairs  

### 🌍 Réseau
- Ajouter ou voir les pairs  
- Synchroniser (consensus)  
- Accéder aux APIs JSON  

### 📜 Historique des blocs
- Liste complète des blocs minés  
- Détails : mineur, nonce, difficulté, transactions, hash  

---

## ⚙️ 5. Installation et utilisation

### Prérequis

- Serveur **PHP 8+**
- Extensions **SQLite** et **OpenSSL** activées
- Navigateur moderne (Chrome, Edge, Firefox, Safari)

---

### Étapes d’installation

1. Copier le dossier `/crypto/` sur votre serveur local ou hébergeur.  
2. Ouvrir `http://localhost/crypto/`  
   (la base `data.db` sera créée automatiquement).  
3. Créer un **compte utilisateur**.  
4. Ajouter des transactions, puis cliquer sur **Miner un bloc**.  
5. Pour le mode P2P :
   - installer la même application sur plusieurs serveurs,  
   - ajouter leurs URL dans la section *Pairs* (ex. `http://localhost:8001/crypto`),  
   - diffuser les transactions et synchroniser.

---

## 🔍 6. Structure de la base SQLite

| Table | Description |
|--------|-------------|
| **users** | Comptes locaux avec clés RSA et adresses |
| **pending** | Transactions en attente de minage |
| **blocks** | Blocs validés (chaîne principale) |
| **peers** | Liste des autres serveurs connectés |

---

## 🔒 7. Limites (volontaires pour l’apprentissage)

- Pas de récompense de minage (*coinbase*)  
- Pas de contrôle de solde (on peut envoyer plus qu’on possède)  
- Pas de chiffrement réseau (HTTP seulement)  
- Pas de consensus distribué complet (simplifié)  
- Clé privée stockée temporairement en session (RAM)

> Ces choix simplifient le code et facilitent la compréhension des concepts fondamentaux.

---

## 🧠 8. Pour aller plus loin (exercices possibles)

1. **Ajouter un contrôle de solde**  
   > Calculer le total reçu – total envoyé pour chaque adresse.  

2. **Récompenser le mineur**  
   > Ajouter une transaction automatique “récompense de minage”.  

3. **Créer un explorateur de blocs**  
   > Page publique listant les blocs et transactions.  

4. **Mesurer le temps de minage**  
   > Comparer les durées selon la difficulté.  

5. **Sécuriser l’API**  
   > Ajouter des clés API ou authentification basique.  

6. **Tester un fork (attaque 51 %)**  
   > Deux serveurs minent des blocs concurrents, puis on résout par consensus.  

7. **Changer le type de signature**  
   > Implémenter Ed25519 ou ECDSA.  

---

## 🎯 9. Résumé visuel

```text
Utilisateur
   ↓
Transaction signée (RSA)
   ↓
Mempool (pending)
   ↓
[ Mineur ]
   ↳ Preuve de travail (nonce → hash “000…”)
   ↓
Bloc validé → ajouté à la chaîne
   ↓
Diffusion aux pairs
   ↓
Consensus : tous adoptent la chaîne la plus longue
```

---

## 💬 10. Remarques finales

- Ce projet est conçu pour **apprendre**, pas pour créer une cryptomonnaie réelle.  
- Il permet de **visualiser les concepts essentiels** : signatures, liens entre blocs, preuve de travail, consensus P2P.  
- Tous les fichiers sont clairs, modulaires et peuvent être étudiés en classe.

---

### 🧾 Auteur
**Projet pédagogique : CryptoGamie**  
Développé pour l’enseignement de la culture numérique et informatique (Genève)  
Langage : PHP 8 / SQLite 3 / Bootstrap 5  
Licence : utilisation libre à but éducatif
