<?php
// auth.php — inscription / connexion / session
require_once __DIR__.'/db.php';
require_once __DIR__.'/crypto.php';
session_start();

function flash(?string $msg = null): ?string {
  if ($msg !== null) { $_SESSION['flash'] = $msg; return null; }
  if (!empty($_SESSION['flash'])) { $m = $_SESSION['flash']; unset($_SESSION['flash']); return $m; }
  return null;
}
function generate_rsa_keys(): array {
    $res = openssl_pkey_new([
        "private_key_bits" => 2048,
        "private_key_type" => OPENSSL_KEYTYPE_RSA,
    ]);

    openssl_pkey_export($res, $privpem);

    $pubDetails = openssl_pkey_get_details($res);
    $pubkey = $pubDetails['key'];

    $address = substr(hash('sha256', $pubkey), 0, 40);

    return [
        'privpem' => $privpem,
        'pubkey'  => $pubkey,
        'address' => $address
    ];
}

function generate_ecdsa_keys(): array {
    // Génère une clé privée EC avec la courbe SECP256k1 (comme Bitcoin)
    $config = [
        "private_key_type" => OPENSSL_KEYTYPE_EC,
        "curve_name"       => "secp256k1"
    ];

    $res = openssl_pkey_new($config);

    // Exporte la clé privée au format PEM
    openssl_pkey_export($res, $privpem);

    // Extrait la clé publique
    $pubDetails = openssl_pkey_get_details($res);
    $pubkey = $pubDetails['key'];

    // Adresse = SHA256(pubkey) tronquée
    $address = substr(hash('sha256', $pubkey), 0, 40);

    return [
        'privpem' => $privpem,
        'pubkey'  => $pubkey,
        'address' => $address
    ];
}

function current_user(): ?array { return $_SESSION['user'] ?? null; }

function register_user($db, $username, $password) {
  if (strlen($username) < 3) {
    throw new Exception("Le nom d'utilisateur doit contenir au moins 3 caractères.");
  }
  if (strlen($password) < 4) {
    throw new Exception("Le mot de passe doit contenir au moins 4 caractères.");
  }

  // Vérifie si le nom est déjà pris avant d'insérer
  $exists = $db->prepare("SELECT COUNT(*) FROM users WHERE username = ?");
  $exists->execute([$username]);
  if ($exists->fetchColumn() > 0) {
    throw new Exception("ce nom d'utilisateur existe déjà. Veuillez en choisir un autre.");
  }

  // Génération des clés ECDSA (SECP256k1) et adresse
  $keys = generate_ecdsa_keys();
  $address = substr(hash('sha256', $keys['pubkey']), 0, 40);
  $hash = password_hash($password, PASSWORD_DEFAULT);

  try {
    $stmt = $db->prepare("
      INSERT INTO users (username, password_hash, pubkey, privpem, address)
      VALUES (?, ?, ?, ?, ?)
    ");
    $stmt->execute([$username, $hash, $keys['pubkey'], $keys['privpem'], $address]);
    return ['username'=>$username, 'address'=>$address];
  } catch (PDOException $e) {
    // Gestion élégante des violations SQL (clé unique, etc.)
    if ($e->getCode() === '23000') {
      throw new Exception("Ce nom d'utilisateur est déjà enregistré. Essayez avec un autre identifiant.");
    } else {
      throw new Exception("Impossible de créer le compte : " . htmlspecialchars($e->getMessage()));
    }
  }
}

function login_user(PDO $db, string $username, string $password): bool {
  $st = $db->prepare("SELECT * FROM users WHERE username=?");
  $st->execute([$username]);
  $u = $st->fetch(PDO::FETCH_ASSOC);

  // Vérifie la présence et le hash correct
  if (!$u || !password_verify($password, $u['password_hash'])) return false;

  // Pas encore de chiffrement actif → on lit la clé privée directement
  $privPemClear = $u['privpem'];

  $_SESSION['user'] = [
    'id'      => $u['id'],
    'username'=> $u['username'],
    'address' => $u['address'],
    'pubkey'  => $u['pubkey'],
    'privpem' => $privPemClear, // PEM privé clair en RAM
  ];
  return true;
}

function logout_user(): void { $_SESSION = []; session_destroy(); }

function post_redirect_self(): void { header("Location: ".$_SERVER['PHP_SELF']); exit; }
