<?php
// chain.php — pending, minage PoW, validation, P2P, consensus
require_once __DIR__.'/db.php';
require_once __DIR__.'/crypto.php';
require_once __DIR__.'/config.php';

// ---------- Pending / Transactions signées ----------
function add_pending_tx(PDO $db, string $sender, string $receiver, float $amount,
                        string $pubPem, string $signature, string $created_at): bool {
  if ($amount <= 0) return false;
  // Adresse cohérente avec la clé publique
  if (address_from_pubkey($pubPem) !== $sender) return false;

  $payload = payload_canonical($sender, $receiver, $amount, $created_at);
  if (!verify_signature($pubPem, $payload, $signature)) return false;

  $payload_hash = hash('sha256', $payload);

  $st = $db->prepare("INSERT OR IGNORE INTO pending
    (sender, receiver, amount, signature, pubkey, created_at, payload_hash)
    VALUES (?,?,?,?,?,?,?)");
  return $st->execute([$sender, $receiver, $amount, $signature, $pubPem, $created_at, $payload_hash]);
}

function get_pending(PDO $db): array {
  return $db->query("SELECT * FROM pending ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
}

function clear_pending(PDO $db): void {
  $db->exec("DELETE FROM pending");
}

// ---------- Blocs / PoW ----------
function last_block(PDO $db): ?array {
  $r = $db->query("SELECT * FROM blocks ORDER BY id DESC LIMIT 1")->fetch(PDO::FETCH_ASSOC);
  return $r ?: null;
}

function pow_mine(string $timestamp, string $tx_json, string $prev_hash, ?string $miner, int $difficulty): array {
  $target = str_repeat('0', $difficulty);
  $nonce = 0;
  do {
    $hash = hash('sha256', $timestamp.$tx_json.$prev_hash.(string)$miner.$nonce);
    $nonce++;
  } while (substr($hash, 0, $difficulty) !== $target);
  return [$nonce-1, $hash];
}

function mine_block(PDO $db, ?string $miner_address, int $difficulty = CHAIN_DIFFICULTY): bool {
  $pending = get_pending($db);
  if (!$pending) return false;

  // (Option pédagogique) vérifier signatures des pending une 2e fois
  foreach ($pending as $t) {
    $payload = payload_canonical($t['sender'], $t['receiver'], floatval($t['amount']), $t['created_at']);
    if (address_from_pubkey($t['pubkey']) !== $t['sender']) return false;
    if (!verify_signature($t['pubkey'], $payload, $t['signature'])) return false;
  }

  $tx_json = json_encode($pending, JSON_UNESCAPED_SLASHES);
  $prev = last_block($db);
  $prev_hash = $prev ? $prev['hash'] : 'GENESIS';
  $timestamp = date('Y-m-d H:i:s');

  [$nonce, $hash] = pow_mine($timestamp, $tx_json, $prev_hash, $miner_address, $difficulty);

  $st = $db->prepare("INSERT INTO blocks
    (timestamp, transactions, prev_hash, hash, nonce, miner, difficulty)
    VALUES (?,?,?,?,?,?,?)");
  $ok = $st->execute([$timestamp, $tx_json, $prev_hash, $hash, $nonce, $miner_address, $difficulty]);
  if ($ok) clear_pending($db);
  return $ok;
}

// ---------- Validation complète ----------
function validate_chain(PDO $db): bool {
  $blocks = $db->query("SELECT * FROM blocks ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
  for ($i=0; $i<count($blocks); $i++) {
    $b = $blocks[$i];
    $target = str_repeat('0', (int)$b['difficulty']);
    $calc = hash('sha256', $b['timestamp'].$b['transactions'].$b['prev_hash'].$b['miner'].$b['nonce']);
    if (substr($calc, 0, (int)$b['difficulty']) !== $target) return false;
    if ($i>0 && $b['prev_hash'] !== $blocks[$i-1]['hash']) return false;

    // Vérifier signatures de toutes les tx
    $txs = json_decode($b['transactions'], true) ?? [];
    foreach ($txs as $t) {
      $payload = payload_canonical($t['sender'], $t['receiver'], floatval($t['amount']), $t['created_at']);
      if (address_from_pubkey($t['pubkey']) !== $t['sender']) return false;
      if (!verify_signature($t['pubkey'], $payload, $t['signature'])) return false;
    }
  }
  return true;
}

// ---------- Peers / P2P ----------
function list_peers(PDO $db): array {
  return $db->query("SELECT * FROM peers ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
}
function add_peer(PDO $db, string $url): bool {
  $url = rtrim($url, '/');
  $st = $db->prepare("INSERT OR IGNORE INTO peers(url) VALUES (?)");
  return $st->execute([$url]);
}

function http_post_json(string $url, array $data): ?array {
  $opts = ['http'=>[
    'method'  => 'POST',
    'header'  => "Content-Type: application/json\r\n",
    'content' => json_encode($data),
    'timeout' => 5
  ]];
  $ctx = stream_context_create($opts);
  $res = @file_get_contents($url, false, $ctx);
  return $res ? json_decode($res, true) : null;
}

function http_get_json(string $url): ?array {
  $ctx = stream_context_create(['http'=>['timeout'=>5]]);
  $res = @file_get_contents($url, false, $ctx);
  return $res ? json_decode($res, true) : null;
}

function broadcast_pending(PDO $db): void {
  $peers = list_peers($db);
  $pool = get_pending($db);
  foreach ($peers as $p) {
    foreach ($pool as $t) {
      http_post_json($p['url'].'/api.php?route=tx_new', [
        'sender'    => $t['sender'],
        'receiver'  => $t['receiver'],
        'amount'    => (float)$t['amount'],
        'signature' => $t['signature'],
        'pubkey'    => $t['pubkey'],
        'created_at'=> $t['created_at']
      ]);
    }
  }
}

// ---------- Consensus (chaîne la plus longue) ----------
function get_chain(PDO $db): array {
  return $db->query("SELECT * FROM blocks ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
}

function validate_external_chain(array $chain): bool {
  for ($i=0; $i<count($chain); $i++) {
    $b = $chain[$i];
    $target = str_repeat('0', (int)$b['difficulty']);
    $calc = hash('sha256', $b['timestamp'].$b['transactions'].$b['prev_hash'].$b['miner'].$b['nonce']);
    if (substr($calc, 0, (int)$b['difficulty']) !== $target) return false;
    if ($i>0 && $b['prev_hash'] !== $chain[$i-1]['hash']) return false;

    $txs = json_decode($b['transactions'], true) ?? [];
    foreach ($txs as $t) {
      $payload = payload_canonical($t['sender'], $t['receiver'], (float)$t['amount'], $t['created_at']);
      if (address_from_pubkey($t['pubkey']) !== $t['sender']) return false;
      if (!verify_signature($t['pubkey'], $payload, $t['signature'])) return false;
    }
  }
  return true;
}

function replace_local_chain(PDO $db, array $new): void {
  $db->beginTransaction();
  $db->exec("DELETE FROM blocks");
  $st = $db->prepare("INSERT INTO blocks(timestamp, transactions, prev_hash, hash, nonce, miner, difficulty)
                      VALUES (?,?,?,?,?,?,?)");
  foreach ($new as $b) {
    $st->execute([
      $b['timestamp'], $b['transactions'], $b['prev_hash'], $b['hash'],
      (int)$b['nonce'], $b['miner'], (int)$b['difficulty']
    ]);
  }
  $db->commit();
}

function resolve_conflicts(PDO $db): bool {
  $best = get_chain($db);
  foreach (list_peers($db) as $p) {
    $r = http_get_json($p['url'].'/api.php?route=chain');
    if (!$r || empty($r['ok'])) continue;
    $remote = $r['blocks'] ?? [];
    if (count($remote) > count($best) && validate_external_chain($remote)) {
      $best = $remote;
    }
  }
  $local = get_chain($db);
  if (count($best) > count($local)) {
    replace_local_chain($db, $best);
    return true;
  }
  return false;
}
