"""
Script utilitaire pour créditer manuellement un utilisateur
dans la blockchain locale (blockchain.db).

À utiliser pour les tests ou pour initialiser un solde.
"""

import sqlite3, time, hashlib, json
from crypto import Block, ChainStore, DB_FILE

def credit_user(address: str, amount: float):
    """Ajoute un bloc contenant une transaction SYSTEM -> utilisateur"""
    store = ChainStore(DB_FILE)

    # récupérer le dernier hash (ou 0 si aucun bloc)
    store.cur.execute("SELECT hash, IFNULL(MAX(index_nb), -1) FROM blocks ORDER BY index_nb DESC LIMIT 1")
    row = store.cur.fetchone()
    previous_hash = row[0] if row and row[0] else "0"
    new_index = (row[1] + 1) if row else 0

    # transaction système
    tx = {
        "sender_pub": "SYSTEM",
        "recipient_address": address,
        "amount": amount,
        "signature_hex": ""
    }

    # créer un bloc très simple (difficulté faible)
    block = Block(
        index=new_index,
        previous_hash=previous_hash,
        timestamp=time.time(),
        transactions=[tx],
        difficulty=1
    )

    # ajouter à la base
    store.add_block(block)
    print(f"Adresse {address} créditée de {amount:.4f} B (bloc #{block.index})")
    print(f"Hash du bloc : {block.hash[:16]}...\n")

def main():
    print("=== Crédit manuel d'un utilisateur ===")
    address = input("Adresse à créditer : ").strip()
    if len(address) < 10:
        print("Adresse invalide.")
        return
    try:
        amount = float(input("Montant à créditer : ").strip())
    except ValueError:
        print("Montant invalide.")
        return
    credit_user(address, amount)

if __name__ == "__main__":
    main()
