"""
sync.py — Synchronisation des utilisateurs et transactions avec le serveur distant parapente.world
Compatibilité : API PHP /data.db (routes user_get, user_register, pool, tx_new, block_add)
Auteur : Antoine Melo / ChatGPT
"""

import urllib.request
import urllib.parse
import json
import base64
import time
import hashlib

# === Configuration serveur distant ===
SERVER_URL = "https://parapente.world/lab/blockchain/main/api.php"

# -------------------------------------------------------------
# === Vérifie la disponibilité du serveur distant ===
# -------------------------------------------------------------
def remote_status():
    try:
        with urllib.request.urlopen(SERVER_URL + "?route=ping", timeout=3) as r:
            return True
    except Exception:
        return False


# -------------------------------------------------------------
# === Récupère un utilisateur par nom ===
# -------------------------------------------------------------
def remote_get_user(username: str):
    try:
        with urllib.request.urlopen(f"{SERVER_URL}?route=user_get&username={username}", timeout=5) as resp:
            data = json.loads(resp.read().decode())
            if not data.get("ok"):
                return None
            u = data.get("user", {})
            # Décodage éventuel de la clé privée
            if "privpem" in u and u["privpem"]:
                try:
                    u["privpem"] = base64.b64decode(u["privpem"]).decode()
                except Exception as e:
                    print("Erreur de décodage privpem:", e)
                    u["privpem"] = ""
            return u
    except Exception as e:
        print("Erreur lors de la récupération utilisateur distant:", e)
    return None


# -------------------------------------------------------------
# === Enregistre un nouvel utilisateur sur le serveur distant ===
# -------------------------------------------------------------
def remote_register_user(username, password_hash, pubkey, privpem, address):
    payload = json.dumps({
        "username": username,
        "password_hash": password_hash,
        "pubkey": pubkey,
        "privpem": base64.b64encode(privpem.encode()).decode(),
        "address": address
    }).encode()
    try:
        req = urllib.request.Request(
            SERVER_URL + "?route=user_register",
            data=payload,
            headers={"Content-Type": "application/json"},
            method="POST"
        )
        with urllib.request.urlopen(req, timeout=5) as resp:
            return json.loads(resp.read().decode())
    except Exception as e:
        print("⚠️ Échec d'enregistrement distant:", e)
        return {"ok": False}


# -------------------------------------------------------------
# === Récupère la pool de transactions distantes ===
# -------------------------------------------------------------
def fetch_remote_pool():
    try:
        with urllib.request.urlopen(SERVER_URL + "?route=pool", timeout=5) as resp:
            data = json.loads(resp.read().decode())
            if data.get("ok"):
                return data.get("pending", [])
    except Exception as e:
        print("Erreur lors du chargement de la pool distante :", e)
    return []


# -------------------------------------------------------------
# === Envoie un bloc miné au serveur distant ===
# -------------------------------------------------------------
def push_mined_block(block_dict):
    try:
        payload = json.dumps({
            "route": "block_add",
            "transactions": block_dict["transactions"],
            "miner": block_dict.get("miner"),
            "prev_hash": block_dict.get("previous_hash"),
            "hash": block_dict.get("hash"),
            "nonce": block_dict.get("nonce"),
            "difficulty": block_dict.get("difficulty", 4)
        }).encode()

        req = urllib.request.Request(
            SERVER_URL + "?route=block_add",
            data=payload,
            headers={"Content-Type": "application/json"},
            method="POST"
        )
        with urllib.request.urlopen(req, timeout=5) as resp:
            return json.loads(resp.read().decode())
    except Exception as e:
        print("Impossible d’envoyer le bloc au serveur :", e)
    return {"ok": False}


# -------------------------------------------------------------
# === Envoie une transaction signée vers le serveur distant ===
# -------------------------------------------------------------
def push_remote_transaction(tx):
    """Publie une transaction locale sur la pool distante."""
    try:
        sender_addr = hashlib.sha256(tx.get("sender_pub", "").encode()).hexdigest()[:40]
        created_at = time.strftime("%Y-%m-%d %H:%M:%S")

        payload = {
            "sender": sender_addr,
            "receiver": tx.get("recipient_address", ""),
            "amount": float(tx.get("amount", 0)),
            "signature": tx.get("signature_hex", ""),
            "pubkey": tx.get("sender_pub", ""),
            "created_at": created_at
        }

        req = urllib.request.Request(
            SERVER_URL + "?route=tx_new",
            data=json.dumps(payload).encode(),
            headers={"Content-Type": "application/json"},
            method="POST"
        )
        with urllib.request.urlopen(req, timeout=5) as resp:
            data = json.loads(resp.read().decode())
            if data.get("ok"):
                print("Transaction publiée sur le serveur distant")
            else:
                print("Échec de publication distante :", data)
    except Exception as e:
        print("Impossible d’envoyer la transaction au serveur :", e)
